//
//  MyLocation.swift
//  Orbis
//
//  Created by KALSI on 5/26/17.
//  Copyright © 2017 KALSI. All rights reserved.
//

import UIKit
import MapKit

class MyLocation: UIViewController, CLLocationManagerDelegate, MKMapViewDelegate {
    
    @IBOutlet weak var myMap: MKMapView!
    @IBOutlet weak var locationLabel: UILabel!
    @IBOutlet weak var addressBG: UIVisualEffectView!
    @IBOutlet weak var crossHairView: UIVisualEffectView!
    @IBOutlet weak var CurrentAddressView: UIView!
    @IBOutlet weak var updatePanel: UIView!
    
    @IBOutlet weak var LastTime: UILabel!
    @IBOutlet weak var LastPlace: UILabel!
    
    var manager = CLLocationManager()
    var myPosition = CLLocationCoordinate2D()
    var isMapDragged = Bool()
    let pin = MKPointAnnotation()
    
    // MARK:-
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(true)
        LastTime.text = String().timeOfLocationUpdate
        LastPlace.text = String().placeOfLocationUpdate
    }
    
    override func viewDidLoad() {
        super.viewDidLoad()
        UserDefaults.standard.set("MyLocation", forKey: "screen?")
        UserDefaults.standard.synchronize()
        isMapDragged = false
        
        //View Attributes
        addressBG.layer.cornerRadius = 8
        addressBG.layer.masksToBounds = true
        Globals.properties().shadow(view: updatePanel, placement: (CGPoint(x: 0, y: 0)), pathType: Constants.kRectangleBezierPath)
        
        //Set Location
        manager.delegate = self
        manager.desiredAccuracy = kCLLocationAccuracyBest
        manager.requestWhenInUseAuthorization()
        manager.startUpdatingLocation()
        myMap.showsUserLocation = true
        
        //Tap Gesture
        let tap = UITapGestureRecognizer(target: self, action: #selector(self.AddAnotation(sender:)))
        tap.numberOfTouchesRequired = 1
        tap.numberOfTapsRequired = 1
        self.myMap.addGestureRecognizer(tap)
    }
    override func didReceiveMemoryWarning() {
        super.didReceiveMemoryWarning()
    }
    //MARK:-
    override func viewWillLayoutSubviews() {
        var Y = (myMap.frame.origin.y + myMap.frame.size.height) - (crossHairView.frame.height + 10)
        let X = (myMap.frame.origin.x + myMap.frame.size.width) - (crossHairView.frame.width + 10)
        crossHairView.frame = CGRect(x:X, y: Y, width: crossHairView.frame.size.width, height: crossHairView.frame.size.width)
        
        Y = myMap.frame.origin.y + 2
//        CurrentAddressView.frame = CGRect(x:CurrentAddressView.frame.origin.x, y: Y, width: CurrentAddressView.frame.size.width, height: CurrentAddressView.frame.size.height)

        crossHairView.layer.cornerRadius = crossHairView.frame.size.width / 2
        crossHairView.layer.masksToBounds = true
    }
    
    //MARK:- Annotation
    func AddAnotation(sender : UIGestureRecognizer) {
        crossHairView.isHidden = false
        let annotationPoint = myMap.convert(sender.location(in: myMap), toCoordinateFrom: myMap)
        if pin.coordinate.latitude != annotationPoint.latitude && pin.coordinate.longitude != annotationPoint.longitude{
            pin.coordinate = annotationPoint
            //            pin.title = "Remove"
            myMap.addAnnotation(pin)
            GetLocationAddressOf(location: annotationPoint)
        }
    }
    
    // MARK: - IBActions -
    @IBAction func CenterMyPosition(_ sender: Any) {
        manager.startUpdatingLocation()
        ShowCurentLocationCentered()
    }
    @IBAction func UpdateLocation(_ sender: Any) {
        
        if !Globals().checkLocationAuthorization(selectedView: self){
            return
        }
        
        
        let updatesDictionary = ["date":String().fullFormatCurrentTime, "location":locationLabel.text!] as [String : Any]
        UserDefaults.standard.setValue(updatesDictionary, forKey: "locationOrb")
        UserDefaults.standard.synchronize()
        
        //Web service
        Globals().SendSignalToARC(controllerObject: self, signalType: Constants.sMapPositionUpdate)
        
        Globals().addPrompt(message: "Location Updated & sent to ARC", view: self) {
            self.LastTime.text = String().timeOfLocationUpdate
            self.LastPlace.text = String().placeOfLocationUpdate
        }
    }
    //MARK:- Location Updates -
    public func locationManager(_ manager: CLLocationManager, didUpdateLocations locations: [CLLocation]){
        myPosition = (manager.location?.coordinate)!
        //show location
        ShowCurentLocationCentered()
        //Get address
        GetLocationAddressOf(location: myPosition)
        manager.stopUpdatingLocation()
    }
    
    func ShowCurentLocationCentered() {
        myMap.removeAnnotations(myMap.annotations)
        let location = CLLocationCoordinate2D(latitude: myPosition.latitude, longitude: myPosition.longitude)
        let span = MKCoordinateSpanMake(0.1, 0.1)
        let region = MKCoordinateRegion(center: location, span: span)
        myMap.setRegion(region, animated: true)
        crossHairView.isHidden = true
    }
    
    func GetLocationAddressOf(location : CLLocationCoordinate2D) {
        let geocoder = CLGeocoder()
        let location = CLLocation(latitude: location.latitude, longitude: location.longitude)
        geocoder.reverseGeocodeLocation(location) { (placemarks, error) in
            if (placemarks != nil) {
                let details = placemarks?[0]
                let placeName = details?.addressDictionary?["Name"] as? String
                self.locationLabel.text = placeName
                self.CurrentAddressView.isHidden = false
            }
            if (error != nil) {
                print(error!)
            }
        }
    }
    
    //MARK:- Map Delegate -
    public func mapView(_ mapView: MKMapView, regionDidChangeAnimated animated: Bool){
        
        crossHairView.isHidden = false
        
        let mapCenterLat = String(format: "\n\n\n map Lat%.5f", mapView.centerCoordinate.latitude)
        let mapCenterLong = String(format: "\n\n\n map Lat%.5f", mapView.centerCoordinate.longitude)
        let myCenterLat = String(format: "\n\n\n map Lat%.5f", myPosition.latitude)
        let myCenterLong = String(format: "\n\n\n map Lat%.5f", myPosition.longitude)
        
        if (mapCenterLat == myCenterLat && mapCenterLong == myCenterLong){
            crossHairView.isHidden = true
        }
    }
    
}

